/**
* \file: AilAudioInImpl.cpp
*
* \version: $Id:$
*
* \release: $Name:$
*
* <brief description>.
* <detailed description>
* \component: CarPlay
*
* \author: J. Michalik / ADIT/SW2 / jmichalik@de.adit-jv.com
*
* \copyright (c) 2016 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
* \see <related items>
*
* \history
*
***********************************************************************/

#include <memory.h>
#include <pthread.h>
#include <sys/prctl.h>
#include <limits.h>
#include <adit_logging.h>
#include <dipo_macros.h>
#include "AilAudioIn.h"
#include "AilAudioInImpl.h"
#include "AilConfiguration.h"

#include "AudioFactory.h"
#include "AudioBackend.h"
#include "AudioTypes.h"

using namespace std;

LOG_IMPORT_CONTEXT(cply);

namespace adit { namespace carplay
{
using namespace adit::utility::audio;
using namespace adit::utility;

AilAudioIn::Impl::Impl()
{
    config = nullptr;
    sink = nullptr;

    processThreadId = 0;

    periodMilli = 0;
    periodSamples = 0;
    appendEmptyPeriods = 0;
    verboseLogging = false;

    running = false;
    std::string backendLibName("Alsa");
    backend = audio::Factory::Instance()->createBackend(backendLibName,*this);
}

AilAudioIn::Impl::~Impl()
{
    /* PRQA: Lint Message 1506: Stop is not expected to be further overridden */
    /*lint -save -e1506*/
    Stop();
    /*lint -restore*/
}

bool AilAudioIn::Impl::Initialize(const IConfiguration& inConfig,
        IAudioInSink& inSink)
{
    config = &inConfig;
    sink = &inSink;

    verboseLogging = AilConfiguration::Instance().IsVerboseLogging(inConfig);

    return true;
}

bool AilAudioIn::Impl::Prepare(AudioFormatStruct inFormat, const std::string& inAudioType)
{
	int32_t buffer_periods = -1; // not necessary to configure AIL
	int32_t silence_ms = -1;     // not necessary to configure AIL
	int32_t inittout_ms = -1;

    (void)inFormat;

    if (config == nullptr)
    {
        LOG_ERROR((cply, "AilAudioOut is not initialized"));
        return false;
    }

    format = inFormat;

    if (format.BitsPerChannel != 24 && format.BitsPerChannel != 16)
    {
        LOG_ERROR((cply, "AilAudioIn does not support %d bit audio", format.BitsPerChannel));
        return false; /* ========== leaving function ========== */
    }

    // numbers of periods to append after end-of-stream
    appendEmptyPeriods = config->GetNumber("alsa-audio-out-append-empty-periods", 0); //ToDo: What is the purpose of this?

    if (!AilConfiguration::Instance().GetDeviceSettings(*config, AilConfiguration::Channel_MainIn,
            inAudioType, inFormat, deviceName /* out */, periodMilli /* out */,
            buffer_periods /* out */, silence_ms /* out */, inittout_ms /* out */))
    {
        LOG_ERROR((cply, "failed to get audio device settings"));
        return false; /* ========== leaving function ========== */
    }
    audio::AudioFormat audioFormat;
    switch (format.BitsPerChannel) {
        case 16 : {
            audioFormat = S16_LE;
            break;
        }
        case 24 : {
            audioFormat = S24_LE;
            break;
        }
        default :
            LOG_ERROR((cply, "AilAudioOut does not support %d bits per channel\n", format.BitsPerChannel));
            return false;
    }

    periodSamples = periodMilli*format.SampleRate/1000;

    std::string playback("");

    if(!config->GetNumber("disable-real-time-priority-audio", 0))
    {
        backend->setThreadSched(SCHED_FIFO, config->GetNumber("audio-threads-real-time-priority", 61));
    }
    if(inittout_ms>0)
    {
        backend->setInitialTimeout(inittout_ms);
    }
    backend->setFadeTime( FadeMode::OUT,  StreamDirection::OUT, 0);
    AudioError err = backend->openStream(deviceName.c_str(), playback , audioFormat, format.SampleRate, format.Channels,
            periodSamples);

    if (err != AudioError::OK) {
        LOG_ERROR((cply, "%s, AilAudioIn::Impl::Prepare(), openStream() failed, error=%d", deviceName.c_str(),
                static_cast<uint32_t>(err)));
        return false;
    } else {
        LOG_INFO((cply, "%s, AilAudioIn::Impl::Prepare(), openStream() success,  periods: %d" , deviceName.c_str(),
                periodSamples));
    }

    LOGD_DEBUG((cply, "main audio in: device=%s, period=%dms",
            deviceName.c_str(), periodMilli));
    return true;
}

bool AilAudioIn::Impl::Start()
{
    running = true;
    sampleNumber = 0;

    AudioError err = backend->startStream();
    if (err != AudioError::OK) {
        LOG_ERROR((cply, "%s, AilAudioIn::Impl::Start(), startStream() failed, error=%d", deviceName.c_str(),
                static_cast<uint32_t>(err)));
        return false;
    } else {
        LOG_INFO((cply, "%s, AilAudioIn::Impl::Start(), startStream() SUCCESS", deviceName.c_str()));
    }
    return true;
}

void AilAudioIn::Impl::Stop()
{
    bool wasRunning = running;
    running = false;

    if(wasRunning) {
        AudioError err = backend->stopStream();
        if (err != AudioError::OK) {
            LOG_ERROR((cply, "%s, stopStream() failed, error=%d", deviceName.c_str(), static_cast<uint32_t>(err)));
        } else {
            LOG_INFO((cply, "%s,  stopStream() SUCCESS", deviceName.c_str()));
        }

        err = backend->closeStream();
        if (err != AudioError::OK) {
            LOG_ERROR((cply, "%s, closeStream() failed, error=%d", deviceName.c_str(), static_cast<uint32_t>(err)));
        } else {
            LOG_INFO((cply, "%s, closeStream() SUCCESS", deviceName.c_str()));
        }
        LOG_INFO((cply, "%s audio in stopped", deviceName.c_str()));
    }
}

void AilAudioIn::Impl::error(const std::string& data) const
{
    LOG_ERROR((cply, "%s, %s",deviceName.c_str(), data.c_str()));
}
void AilAudioIn::Impl::warning(const std::string& data) const
{
    LOG_WARN((cply, "%s, %s",deviceName.c_str(), data.c_str()));
}
void AilAudioIn::Impl::info(const std::string& data) const
{
    LOG_INFO((cply, "%s, %s",deviceName.c_str(), data.c_str()));
}
void AilAudioIn::Impl::debug(const std::string& data) const
{
    LOGD_DEBUG((cply, "%s, %s",deviceName.c_str(), data.c_str() ));
}
eLogLevel AilAudioIn::Impl::checkLogLevel() const
{
    return eLogLevel::LL_MAX;
}

AudioState AilAudioIn::Impl::processing(unsigned char *in, unsigned char **out, uint32_t &frames)
{
    (void) out;
    const uint32_t numOfBytesPerSample = format.BitsPerChannel/8;

    samples.Length = frames* (numOfBytesPerSample) * format.Channels;
    samples.TimeStamp = sampleNumber;
    samples.DataPtr = in;

    sink->Write(samples);
    sampleNumber += periodSamples;

    if (verboseLogging) {
        LOGD_DEBUG((cply, "%s Data process - running frames%zd DataPtr%d",deviceName.c_str(), samples.Length,
                *(uint32_t*)(samples.DataPtr)));
    }

    return AudioState::CONTINUE;
}
void AilAudioIn::Impl::statistics(const StreamStatistics &status)
{
    (void) status;
}
void AilAudioIn::Impl::eostreaming(const AudioError error)
{
    if (error != AudioError::OK) {
        LOG_ERROR((cply, "%s, eostreaming(): Streaming has stopped unexpectedly: %d",deviceName.c_str(),
                (uint32_t)error));
    }
}

} } // namespace adit { namespace carplay
